<?php

declare(strict_types=1);

namespace Codeception\Lib\Connector\Laravel;

use Illuminate\Contracts\Debug\ExceptionHandler as ExceptionHandlerContract;
use Illuminate\Http\Request;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\HttpFoundation\Response;
use Throwable;
use function strpos;

class ExceptionHandlerDecorator implements ExceptionHandlerContract
{
    private ExceptionHandlerContract $laravelExceptionHandler;

    private bool $exceptionHandlingDisabled = true;

    public function __construct(ExceptionHandlerContract $exceptionHandler)
    {
        $this->laravelExceptionHandler = $exceptionHandler;
    }

    public function exceptionHandlingDisabled(bool $exceptionHandlingDisabled): void
    {
        $this->exceptionHandlingDisabled = $exceptionHandlingDisabled;
    }

    /**
     * Report or log an exception.
     *
     * @param Throwable $e
     * @throws Throwable
     */
    public function report(Throwable $e): void
    {
        $this->laravelExceptionHandler->report($e);
    }

    /**
     * Determine if the exception should be reported.
     */
    public function shouldReport(Throwable $e): bool
    {
        return $this->exceptionHandlingDisabled;
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param Request $request
     * @throws Throwable
     */
    public function render($request, Throwable $e): Response
    {
        $response = $this->laravelExceptionHandler->render($request, $e);

        if ($this->exceptionHandlingDisabled && $this->isSymfonyExceptionHandlerOutput($response->getContent())) {
            // If content was generated by the \Symfony\Component\Debug\ExceptionHandler class
            // the Laravel application could not handle the exception,
            // so re-throw this exception if the Codeception user disabled Laravel exception handling.
            throw $e;
        }

        return $response;
    }

    /**
     * Check if the response content is HTML output of the Symfony exception handler class.
     */
    private function isSymfonyExceptionHandlerOutput(string $content): bool
    {
        return strpos($content, '<div id="sf-resetcontent" class="sf-reset">') !== false ||
            strpos($content, '<div class="exception-summary">') !== false;
    }

    /**
     * Render an exception to the console.
     *
     * @param  OutputInterface $output
     */
    public function renderForConsole($output, Throwable $e): void
    {
        $this->laravelExceptionHandler->renderForConsole($output, $e);
    }

    /**
     * @param string|callable $method
     */
    public function __call($method, array $args)
    {
        return call_user_func_array([$this->laravelExceptionHandler, $method], $args);
    }
}
